#pragma module DirectoryService "V1.0-00"

#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <ldap.h>

#include "DirectoryService.h"


NamingContext	*ndi_ldap_initiateLDAP( const char *server, const char *sb );
void ndi_ldap_doContextSetup( NamingContext *dsi );
void ndi_ldap_get_optional_message( NamingContext *dsi );

typedef unsigned char	UBYTE;


//
// NOTE ! The search base string is always appended to the search string given
//	  for every search. If search base is O=SEB and You specify CN=Jonny, the resulting search
//	  would be CN=Jonny,O=SEB else it would be only O=SEB

//
// NOTE ! All returned strings are dynamically allocated and must be freed
//	  All returned AttributeInfo structures must be freed using the
//	  ndi_ldap_freeAttributeInfo( ) function

//
//	ndi_ldap_Create
//
//	Creates the needed internal datastructures needed for the handling of
//	user requests. No call to the LDAP service is actually done
//
//	Args:	sb	String containing the specified search base
//		server	String containing the name of the ldap server host
//			like SEB500 or SEB500.SEBANK.SE
//
//	Return:	dsi	Pointer to the NamingContext area
//
//	Status handling by the ndi_ldap_getStatus( )
//
NamingContext	*ndi_ldap_Create( const char *sb, const char *server )
	{
	return	ndi_ldap_EnableTrace( sb, server, FALSE );
	}					// End of constructor ndi_ldap_Create( )

//
//	ndi_ldap_SetTraceFile
//
//	Creates the needed internal datastructures needed for the handling of
//	user requests. No call to the LDAP service is actually done
//	Messages from the DirectoryService methods are written to the specified file
//	which must be opened & closed by the user
//
//	Args:	sb	String containing the specified search base
//		server	String containing the name of the ldap server host
//			like SEB500 or SEB500.SEBANK.SE
//		lto	Pointer to a  file opend for output
//
//	Return:	dsi	Pointer to the NamingContext area
//
//	Status handling by the ndi_ldap_getStatus( )
//
NamingContext	*ndi_ldap_SetTraceFile( const char *sb, const char *server, FILE *lto )
	{
NamingContext	*dsi	= NULL;

	dsi	= ndi_ldap_initiateLDAP( server, sb );

	if ( dsi != NULL )
	    {
	    if ( lto != NULL )
		{
		dsi->ldapTrace_	= lto;
		}

	    ndi_ldap_doContextSetup( dsi );
	    }					// if ( dsi != NULL )

	return	dsi;
	}					// End of constructor ndi_ldap_SetTraceFile( )

//
//	ndi_ldap_EnableTrace
//
//	Creates the needed internal datastructures needed for the handling of
//	user requests. No call to the LDAP service is actually done
//	Trace & error messages may be written to stderr
//
//	Args:	sb	String containing the specified search base
//		server	String containing the name of the ldap server host
//			like SEB500 or SEB500.SEBANK.SE
//		tflag	Boolean indicating that trace should or should not
//			be performed to stderr
//
//	Return:	dsi	Pointer to the NamingContext area
//
//	Status handling by the ndi_ldap_getStatus( )
//
NamingContext	*ndi_ldap_EnableTrace( const char *sb, const char *server, BOOLEAN tflag )
	{
NamingContext	*dsi	= NULL;

	dsi	= ndi_ldap_initiateLDAP( server, sb );

	if ( dsi != NULL )
	    {
	    if ( tflag )
		{
		dsi->ldapTrace_	= stderr;
		}

	    ndi_ldap_doContextSetup( dsi );
	    }					// if ( dsi != NULL )

	return	dsi;
	}					// End of constructor ndi_ldap_EnableTrace( )

//
//	ndi_ldap_initiateLDAP	( INTERNAL ONLY )
//
//	Allocates the internal datastructures and loads it with the
//	required setup parameters
//
//	Args:	sb	String containing the specified search base
//		server	String containing the name of the ldap server host
//			like SEB500 or SEB500.SEBANK.SE
//
//	Return:	dsi	Pointer to the service area or null
//
//	Status handling by the ndi_ldap_getStatus( )
//
NamingContext	*ndi_ldap_initiateLDAP( const char *server, const char *sb )
	{

int	stat		= -1;

NamingContext	*dsi;

	dsi	= malloc( sizeof( NamingContext ) );

	if ( dsi == NULL )
	    {
	    return	dsi;
	    }

	memset( dsi, 0, sizeof( NamingContext ) );

	dsi->Bound_		= FALSE;
	dsi->ldapCtx_		= NULL;
	dsi->searchBase_	= NULL;
	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;
	dsi->ldapProtocol	= LDAP_VERSION3;

	dsi->ldapServer_	= strdup( server );
	dsi->searchBase_	= strdup( sb );

	dsi->ldapTrace_		= NULL;

	dsi->ldapStatusMsg1_	= malloc( MESSAGE_AREA_SIZE );
	dsi->ldapStatusMsg2_	= malloc( MESSAGE_AREA_SIZE );

	strcpy( dsi->ldapServer_, server );
	strcpy( dsi->searchBase_, sb );

	ndi_ldap_str_to_upper( dsi->ldapServer_ ),
	ndi_ldap_str_to_upper( dsi->searchBase_ ),

	sprintf( dsi->ldapStatusMsg1_, "Directory Service is not initiated" );

	dsi->ldapCtx_	=  ldap_init( dsi->ldapServer_, LDAP_PORT );

	if ( dsi->ldapCtx_ == NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_INIT_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Initiating the Directory Service at %s failed", dsi->ldapServer_ );

	    return	dsi;
	    }

	//
	// OK, now we have been provided with a LDAP context, it is possible now to
	// specify options for this session
	//

	stat = ldap_set_option( dsi->ldapCtx_, LDAP_OPT_PROTOCOL_VERSION, &dsi->ldapProtocol );

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_INIT_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "ldap_set_option returned error %d ( '%s' )", stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    return	dsi;
	    }

	return	dsi;
	}					// End of initateLDAP( )

//
//	doContextSetup	( INTERNAL ONLY )
//
//	A dummy module to keep the similarity with Java
//
//	Args:	NONE
//
//	Return:	NONE
//
//	Status handling by the ndi_ldap_getStatus( )
//
void	ndi_ldap_doContextSetup( NamingContext *dsi )
	{
	dsi->ldapStatus_	= RS_LDAP_OK;

	return;
	}					// End of ndi_ldap_doContextSetup( )

//
//	simpleBind
//
//	The method acts as an connect/reconnect to the LDAP Service.
//	It handles calls for the LDAP bind function and authentication
//	The uid should be fully distinguished and pwd is case sensitive
//	The uid and / or the pwd may be null 
//
//	Args:	uid	String containing the DN to bind with
//		pwd	String containing the password for the DN
//
//	Return:	True	Bind was succesfull
//		False	Bind failed
//
//	Status handling by the ndi_ldap_getStatus( )
//
BOOLEAN ndi_ldap_simpleBind( NamingContext *dsi, const char *uid, const char *pwd )
	{
int	stat	= -1;

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	if (( uid != NULL ) && ( strlen( uid ) == 0 ))
	    {
	    dsi->ldapStatus_	= RS_LDAP_INVALID_UID;

	    sprintf( dsi->ldapStatusMsg1_, "Invalid user identity for simple bind" );

	    return	FALSE;
	    }					// if ( uid == NULL )

	if (( pwd != NULL ) && ( strlen( pwd ) == 0 ))
	    {
	    dsi->ldapStatus_	= RS_LDAP_INVALID_PWD;

	    sprintf( dsi->ldapStatusMsg1_, "Invalid user password for simple bind" );

	    return	FALSE;
	    }					// if ( pwd == NULL )

	//
	// Do a bind setup
	//

	stat = ldap_simple_bind_s( dsi->ldapCtx_, uid, pwd );

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_BIND_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Bind failed with error %d ( '%s' ) , user ID : '%s'", stat, ldap_err2string( stat ), uid );

	    ndi_ldap_get_optional_message( dsi );

	    dsi->Bound_	= FALSE;

	    return	FALSE;
	    }

	dsi->Bound_		= TRUE;
	dsi->ldapStatus_	= RS_LDAP_OK;

	return	TRUE;
        }					// End of ndi_ldap_simpleBind( )
     

//
//	ndi_ldap_getListByDN
//
//	Search for a specific entry using a specified DN like CN=Fullname,O=SEB and
//	retrieve all available attributes as a list
//	It returns all attributes found ( except the password attribute ), operational
//	attributes are not returned
//
//	Args:	nam_s	String containing the DN to get attributes for
//
//	Return:		Pointer to an Attribute Info array
//			If nothing found a NULL is returned or if any error
//
//	Status handling by the ndi_ldap_getStatus( )
//
AttributeInfo **ndi_ldap_getListByDN( NamingContext *dsi, const char *nam_s )
	{

int		stat		= -1;
int		idx		= 0;
int		num_entries	= 0;

LDAPMessage	*result		= NULL;
LDAPMessage	*one_entry	= NULL;

char		*attr_name	= NULL;

BerElement	*ber_element	= NULL;
BerValue	**attr_value	= NULL;

AttributeInfo	**res_a		= NULL;

AttributeInfo	*attribute	= NULL;
char		*res_s		= NULL;

DN_Area		newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	NULL;
	    }

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy( newname_s, nam_s );
	    strcat(( char * ) newname_s, "," );
	    strcat(( char * ) newname_s, dsi->searchBase_ );
	    }

	stat = ldap_search_s( dsi->ldapCtx_,
				( char * ) &newname_s,
				LDAP_SCOPE_SUBTREE,	// Scope
				NULL,			// Filter
				NULL,			// Attributes
				FALSE,			// Only types
				&result );		// Search result

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attributes for : '%s' failed with error %d ( '%s' )", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    return NULL;
	    }

	// if nat has more than one argument then something is wrong,
	// the arguments will not give ONE hit which is required

	num_entries = ldap_count_entries( dsi->ldapCtx_, result );

	if ( num_entries == 0 )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attributes for : '%s' failed with error %d ( '%s' ) ", newname_s, stat, ldap_err2string( stat ));

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		result	= NULL;
		}  

	    return NULL;
	    }					// If ( num_entries == 0 )

	if ( num_entries > 1 )
	    {
	    // If there exits more elements, then return NULL since it was a bad match

	    dsi->ldapStatus_	= RS_LDAP_SEARCH_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Search returned ambiguous result of : '%s' - more than one match", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }

	// OK, check the result

	one_entry	= ldap_first_entry( dsi->ldapCtx_, result );

	attr_name	= ldap_first_attribute( dsi->ldapCtx_, one_entry, &ber_element );

	// OK, loop and get the attribute name and value
	// But first allocate a return structure
	//
	// Allocate enough room for a pointer to the return attributes, plus
	// a NULL terminating pointer. 
	//

	res_a	= ( AttributeInfo ** ) calloc(( MAX_ATTRIBUTES + 1), sizeof( AttributeInfo * ) );

	idx	= 0;
	while ( attr_name != NULL )
	    {
	    attribute	= malloc( sizeof( AttributeInfo ));
	    memset( attribute, 0, sizeof( AttributeInfo ));

	    res_a[ idx++ ]	= attribute;

	    attribute->t_val	= strdup( attr_name );

	    attr_value	= ldap_get_values_len( dsi->ldapCtx_, one_entry, attr_name );

	    // Check if attribute has a value, the password attribute
	    // does not have one, then return an empty object, not NULL

	    if ( attr_value != NULL )
		{

		res_s	= malloc(( attr_value[ 0 ]->bv_len ) + 2 );
		memcpy( res_s, attr_value[ 0 ]->bv_val, attr_value[ 0 ]->bv_len );

		attribute->v_len	= attr_value[ 0 ]->bv_len;
		attribute->v_val	= res_s;
		}				// if ( attr_value == NULL )

	    ldap_value_free_len( attr_value );

	    attr_value	= NULL;

	    ldap_memfree( attr_name );

	    attr_name	= NULL;

	    ldap_value_free_len( attr_value );

	    attr_value	= NULL;

	    attr_name	= ldap_next_attribute( dsi->ldapCtx_, one_entry, ber_element );
	    }					// while ( attr_name != NULL )

	dsi->ldapStatus_	= RS_LDAP_OK;

	if ( result != NULL )
	    {
	    ldap_msgfree( result );
	    }  

	if ( ber_element != NULL )
	    {
	    ber_free( ber_element, 0 );
	    }					// if ( ber_element != NULL )

	return	res_a;
	}					// End of ndi_ldap_getListByDN( )

//
//	ndi_ldap_getDNFromAttr
//
//	Search for a specific entry using a list of attributes containing
//	type & value pairs. This is mainly used to get the DN for a entry
//	which is to be used later in calls.
//
//	Args:	nam_s	String with additions to the search base, normaly len = 0
//		bas_a	AttributeInfo array containing the attributes to
//			find the DN for
//
//	Return:		Pointer to a string with the DN if found
//			NULL if any error
//
//	Status handling by the ndi_ldap_getStatus( )
//
char *ndi_ldap_getDNFromAttr( NamingContext *dsi, const char *nam_s, AttributeInfo **bas_a )
	{
int		idx		= 0;
int		stat		= -1;
int		num_entries	= 0;

LDAPMessage	*result		= NULL;
LDAPMessage	*one_entry	= NULL;

AttributeInfo	*attribute	= NULL;

char		**sea_a		= NULL;

char		*dn		= NULL;
char		*res_s		= NULL;

char		*filter		= NULL;

DN_Area		newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	NULL;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy( newname_s, nam_s );
	    strcat(( char * ) newname_s, "," );
	    strcat(( char * ) newname_s, dsi->searchBase_ );
	    }

	// Create the filter string

	filter	= malloc( FILTER_AREA_SIZE );

	memset( filter, 0, FILTER_AREA_SIZE );

	strcat( filter, "(&" );
	//
	// Add the attributes to filter string
	//
	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    attribute	= bas_a[ idx++ ];

	    strcat( filter, "(" );
	    strcat( filter, attribute->t_val );
	    strcat( filter, "=" );
	    strcat( filter, attribute->v_val );
	    strcat( filter, ")" );
	    }					// while ( bas_a[ ] )

	strcat( filter, ")" );

	// And do the search

	stat = ldap_search_s( dsi->ldapCtx_,
				( char * ) &newname_s,
				LDAP_SCOPE_SUBTREE,	// Scope
				filter,			// Filter
				sea_a,			// Attributes
				FALSE,			// Only types
				&result );		// Result

	free( filter );

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_SEARCH_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of DN using attributes in : '%s' failed with error %d ( '%s' ) ", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    return NULL;
	    }

	// if result has more than one entry then something is wrong,
	// the arguments will not give ONE hit which is required

	num_entries = ldap_count_entries( dsi->ldapCtx_, result );

	if ( num_entries == 0 )
	    {
	    dsi->ldapStatus_	= RS_LDAP_SEARCH_FAIL;

	    sprintf( "Retrieve of DN using attributes in : '%s' failed with error %d ( '%s' ) ", newname_s, stat, ldap_err2string( stat ));

	    return NULL;
	    }					// If ( num_entries == 0 )

	if ( num_entries > 1 )
	    {
	    // If there exits more elements, then return NULL since it was a bad match

	    dsi->ldapStatus_	= RS_LDAP_SEARCH_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Search returned ambiguous result of : '%s' - more than one match", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }

	// OK, check the result

	one_entry	= ldap_first_entry( dsi->ldapCtx_, result );

	dn		= ldap_get_dn( dsi->ldapCtx_, one_entry );

	res_s	= malloc( strlen( dn ) + 10 );
	memset( res_s, 0, strlen( dn ) + 10 );

	strcpy( res_s, dn );

	ldap_memfree( dn );

	if ( result != NULL )
	    {
	    ldap_msgfree( result );
	    }  

	dsi->ldapStatus_	= RS_LDAP_OK;

	// Ok, return the DN string pointer

	return	res_s;
	}					// End of ndi_ldap_getDNFromAttr( )

//
//	ndi_ldap_getAttrValueByDN
//
//	Retrieves the value for a specified attribute in a specified entry
//	using the DN like CN=Fullname,O=SEB.
//
//	NOTE !	If the attribute search requested returns a valid result
//		but not any attribute value like the password attribute,
//		an empty attribute value is returned
//	NOTE ! This works only with nonbinary strings
//
//	Args:	nam_s	String with a DN without the defined searchbase
//		attr_s	String containing the attribute to return a value for
//
//	Return:		Pointer to a string with the attribute value if found
//			NULL if any error
//
//	Status handling by the ndi_ldap_getStatus( )
//
char *ndi_ldap_getAttrValueByDN( NamingContext *dsi, const char *nam_s, const char *attr_s )
	{
int		stat		= -1;
int		num_entries	= 0;

LDAPMessage	*result		= NULL;
LDAPMessage	*one_entry	= NULL;

char		**sea_a		= NULL;

char		*attr_name	= NULL;

BerElement	*ber_element	= NULL;
BerValue	**attr_value	= NULL;

char		*res_s		= NULL;

DN_Area		newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	NULL;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	//
	// Allocate enough room for a pointer to the attribute, plus
	// a NULL terminating pointer. 
	//

	sea_a	= ( char ** ) calloc(( 1 + 1), sizeof( char * ) );

	sea_a[ 0 ]	= strdup( attr_s );

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy( newname_s, nam_s );
	    strcat(( char * ) newname_s, "," );
	    strcat(( char * ) newname_s, dsi->searchBase_ );
	    }

	stat = ldap_search_s( dsi->ldapCtx_,
				( char * ) &newname_s,
				LDAP_SCOPE_SUBTREE,	// Scope
				NULL,			// Filter
				sea_a,			// Attributes
				FALSE,			// Only types
				&result );		// Search result

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attributes for : '%s' failed, error %d (%s)", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    return NULL;
	    }

	// if nat has more than one argument then something is wrong,
	// the arguments will not give ONE hit which is required

	num_entries = ldap_count_entries( dsi->ldapCtx_, result );

	if ( num_entries == 0 )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attributes for : '%s' failed, not found", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return NULL;
	    }					// If ( num_entries == 0 )

	if ( num_entries > 1 )
	    {
	    // If there exits more elements, then return NULL since it was a bad match

	    dsi->ldapStatus_	= RS_LDAP_SEARCH_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Search returned ambiguous result of : '%s' - more than one match", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }

	// OK, check the result

	one_entry	= ldap_first_entry( dsi->ldapCtx_, result );

	attr_name	= ldap_first_attribute( dsi->ldapCtx_, one_entry, &ber_element );

	if ( attr_name == NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_NO_ATTR;

	    sprintf( dsi->ldapStatusMsg1_, "The specified attribute was not found for : '%s' - select attribute is '%s'", newname_s, attr_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }					// if ( attr_name == NULL )

	attr_value	= ldap_get_values_len( dsi->ldapCtx_, one_entry, attr_name );

	// Check if attribute has a value, the password attribute
	// does not have one, then return an empty object, not NULL

	res_s	= malloc(( attr_value[ 0 ]->bv_len ) + 2 );
	memset( res_s, 0, ( attr_value[ 0 ]->bv_len ) + 2 );

	if ( attr_value != NULL )
	    {
	    memcpy( res_s, attr_value[ 0 ]->bv_val, attr_value[ 0 ]->bv_len );
	    }				// if ( attr_value == NULL )

	dsi->ldapStatus_	= RS_LDAP_OK;

	ldap_value_free_len( attr_value );

	attr_value	= NULL;

	ldap_memfree( attr_name );

	attr_name	= NULL;

	attr_name	= ldap_next_attribute( dsi->ldapCtx_, one_entry, ber_element );

	// if we now have more attributes then we are in trouble, return
	// a NULL object instead

	if ( attr_name != NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_ATTR_AMBIGUOUS;

	    sprintf( "Ambiguous attributes returned for : '%s' - select attribute is '%s'", newname_s, attr_s );

	    free( res_s );

	    res_s	= NULL;
	    }

	ldap_value_free_len( attr_value );

	attr_value	= NULL;

	if ( result != NULL )
	    {
	    ldap_msgfree( result );
	    }  

	if ( ber_element != NULL )
	    {
	    ber_free( ber_element, 0 );
	    }					// if ( ber_element != NULL )

	return	res_s;
	}					// End of ndi_ldap_getAttrValueByDN( )

//
//	ndi_ldap_getAttrByDN
//
//	Get a attribute structure containing the type & value pair using
//	a DN like CN=Fullname,O=SEB. Works also with binary values.
//
//	Args:	nam_s	String with a DN without the defined searchbase
//		attr_s	String containing the attribute to return
//
//	Return:		Pointer to an AttributeInfo structure
//			NULL if any error
//
//	Status handling by the ndi_ldap_getStatus( )
//
AttributeInfo *ndi_ldap_getAttrByDN( NamingContext *dsi, const char *nam_s, const char *attr_s )
	{
int		stat		= -1;
int		num_entries	= 0;

LDAPMessage	*result		= NULL;
LDAPMessage	*one_entry	= NULL;

char		**sea_a		= NULL;

char		*attr_name	= NULL;

BerElement	*ber_element	= NULL;
BerValue	**attr_value	= NULL;

AttributeInfo	*attribute	= NULL;

char		*res_s		= NULL;

DN_Area		newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	NULL;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	//
	// Allocate enough room for a pointer to the attribute, plus
	// a NULL terminating pointer. 
	//

	sea_a	= ( char ** ) calloc(( 1 + 1), sizeof( char * ) );

	sea_a[ 0 ]	= strdup( attr_s );

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy(( char * ) &newname_s, nam_s );
	    strcat(( char * ) &newname_s, "," );
	    strcat(( char * ) &newname_s, dsi->searchBase_ );
	    }

	stat = ldap_search_s( dsi->ldapCtx_,
				( char * ) &newname_s,
				LDAP_SCOPE_SUBTREE,	// Scope
				NULL,			// Filter
				sea_a,			// Attributes
				FALSE,			// Only types
				&result );		// Search result

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attributes for : '%s' failed, error %d (%s) ", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    return NULL;
	    }

	// if nat has more than one argument then something is wrong,
	// the arguments will not give ONE hit which is required

	num_entries = ldap_count_entries( dsi->ldapCtx_, result );

	if ( num_entries == 0 )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attributes for : '%s' failed", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return NULL;
	    }					// If ( num_entries == 0 )

	if ( num_entries > 1 )
	    {
	    // If there exits more elements, then return NULL since it was a bad match

	    dsi->ldapStatus_	= RS_LDAP_SEARCH_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Search returned ambiguous result of : '%s' - more than one match", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }

	// OK, check the result

	one_entry	= ldap_first_entry( dsi->ldapCtx_, result );

	attr_name	= ldap_first_attribute( dsi->ldapCtx_, one_entry, &ber_element );

	if ( attr_name == NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_NO_ATTR;

	    sprintf( dsi->ldapStatusMsg1_, "The specified attribute was not found for : '%s' - select attribute is '%s'", newname_s, attr_s );

	   if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }					// if ( attr_name == NULL )

	attribute	= malloc( sizeof( AttributeInfo ));
	memset( attribute, 0, sizeof( AttributeInfo ));

	attribute->t_val	= strdup( attr_name );

	attr_value	= ldap_get_values_len( dsi->ldapCtx_, one_entry, attr_name );

	// Check if attribute has a value, the password attribute
	// does not have one, then return an empty object, not NULL

	if ( attr_value != NULL )
	    {

	    res_s	= malloc(( attr_value[ 0 ]->bv_len ) + 2 );
	    memcpy( res_s, attr_value[ 0 ]->bv_val, attr_value[ 0 ]->bv_len );

	    attribute->v_len	= attr_value[ 0 ]->bv_len;
	    attribute->v_val	= res_s;
	    }				// if ( attr_value == NULL )

	dsi->ldapStatus_	= RS_LDAP_OK;

	ldap_value_free_len( attr_value );

	attr_value	= NULL;

	ldap_memfree( attr_name );

	attr_name	= NULL;

	attr_name	= ldap_next_attribute( dsi->ldapCtx_, one_entry, ber_element );

	// if we now have more attributes then we are in trouble, return
	// a NULL object instead

	if ( attr_name != NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_ATTR_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Ambiguous attributes returned for : '%s' - select attribute is '%s'", newname_s, attr_s );

	    ldap_memfree( attr_name );

	    attr_name	= NULL;

	    free( attribute->t_val );
	    free( attribute->v_val );
	    free( attribute );

	    attribute	= NULL;
	    }

	ldap_value_free_len( attr_value );

	if ( result != NULL )
	    {
	    ldap_msgfree( result );
	    }  

	if ( ber_element != NULL )
	    {
	    ber_free( ber_element, 0 );
	    }					// if ( ber_element != NULL )

	return	attribute;
	}					// End of ndi_ldap_getAttrByDN( )

//
//	ndi_ldap_addEntryWithAttrs( )
//
//	Adds ( creates ) a new entry using a DN like CN=Fullname,O=SEB and
//	a list of attribute pairs. All attribute pairs must be set up by
//	the user with the correct values.
//
//	Args:	nam_s	String with a DN without the defined searchbase
//		bas_a	AttributeInfo structure containing ALL attributes
//			for the entry
//
//	Return:	TRUE	Entry was added
//		FALSE	The entry was not added
//
//	Status handling by the ndi_ldap_getStatus( )
//
BOOLEAN ndi_ldap_addEntryWithAttrs( NamingContext *dsi, const char *nam_s, AttributeInfo **bas_a )
	{
int		stat		= -1;
int		idx		= 0;
int		midx		= 0;
int		vidx		= 0;

BOOLEAN		ret_status	= TRUE;

LDAPMod		**attr_array	= NULL;
LDAPMod		*mod_attr	= NULL;

char		**values	= NULL;

AttributeInfo	*attribute	= NULL;

DN_Area	newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	FALSE;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	attr_array	= ( LDAPMod ** ) calloc( MAX_ATTRIBUTES + 1, sizeof( char * ));

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy(( char * ) &newname_s, nam_s );
	    strcat(( char * ) &newname_s, "," );
	    strcat(( char * ) &newname_s, dsi->searchBase_ );
	    }

	//
	// Create the attribute insertion array
	//
	midx	= 0;
	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    mod_attr	= malloc( sizeof( LDAPMod ));
	    memset( mod_attr, 0, sizeof( LDAPMod ));

	    attribute	= bas_a[ idx++ ];

	    mod_attr->mod_op	= LDAP_MOD_ADD;
	    mod_attr->mod_type	= attribute->t_val;

	    values	= ( char ** ) calloc( 50 + 1, sizeof( char * ));
	    vidx	= 0;

	    values[ vidx++ ]	= attribute->v_val;

	    values[ vidx ]	= NULL;

	    mod_attr->mod_vals.modv_strvals	= values;

	    attr_array[ midx++ ]	= mod_attr;
	    }					// while ( bas_a[ ] )

	attr_array[ midx ]	= NULL;

	stat	= ldap_add_s(	dsi->ldapCtx_,
				( char * ) &newname_s,
				attr_array );
    
	dsi->ldapStatus_	= RS_LDAP_OK;

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_ENTRY_EXISTS;

	    sprintf( dsi->ldapStatusMsg1_, "Entry '%s' already exists, error %d (%s) ", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    ret_status	= FALSE;
	    }
	//
	// Free the attribute list
	//
	idx	= 0;
	while ( attr_array[ idx ] != NULL )
	    {
	    mod_attr	= attr_array[ idx ];

	    free( mod_attr->mod_type );
	    free( mod_attr->mod_vals.modv_strvals );

	    free( mod_attr );

	    attr_array[ idx++ ]	= NULL;
	    }					// while( attr_array[ idx ] != NULL )

	free( attr_array );

	return	ret_status;
	}					// End of ndi_ldap_addEntryWithAttrs( )

//
//	ndi_ldap_removeEntryByDN
//
//	Removes an entry using a DN like CN=Fullname,O=SEB. Subentries
//	will also be deleted.
//
//	Args:	nam_s	String with a DN without the defined searchbase
//
//	Return:	TRUE	Entry was removed
//		FALSE	The entry was not removed
//
BOOLEAN ndi_ldap_removeEntryByDN( NamingContext *dsi, const char *nam_s )
	{
int	stat	= -1;

BOOLEAN	ret_status	= TRUE;

DN_Area	newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	FALSE;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy(( char * ) &newname_s, nam_s );
	    strcat(( char * ) &newname_s, "," );
	    strcat(( char * ) &newname_s, dsi->searchBase_ );
	    }

	stat	= ldap_delete_s(	dsi->ldapCtx_,
					( char * ) &newname_s );

	dsi->ldapStatus_	= RS_LDAP_OK;

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_NO_ENTRY;

	    sprintf( dsi->ldapStatusMsg1_, "Removal of entry '%s' failed, does not exists, error %d (%s)", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    ret_status	= FALSE;
	    }
    
	return	ret_status;
	}					// End of ndi_ldap_removeEntryByDN( )

//
//	ndi_ldap_modifyAttrByDN
//
//	Modifies an attribute for a specified entry using a DN and
//	a list of Attributes to be modified. Valid attributes not already
//	in entry is added to the entry.
//
//	Args:	nam_s	String with a DN without the defined searchbase
//		bas_a	AttributeInfo structure containing attributes
//			to modify for the entry
//
//	Return:	TRUE	Entry was added
//		FALSE	The entry was not added
//
BOOLEAN ndi_ldap_modifyAttrByDN( NamingContext *dsi, const char *nam_s, AttributeInfo **bas_a )
	{
int		stat		= -1;
int		midx		= 0;
int		idx		= 0;
int		vidx		= 0;

BOOLEAN		ret_status	= TRUE;

LDAPMod		**attr_array	= NULL;
LDAPMod		*mod_attr	= NULL;

char		**values	= NULL;

AttributeInfo	*attribute	= NULL;

DN_Area		newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	FALSE;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	attr_array	= ( LDAPMod ** ) calloc(( strlen(( char * ) *bas_a ) / sizeof( char * )) + 1, sizeof( char * ));

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy(( char * ) &newname_s, nam_s );
	    strcat(( char * ) &newname_s, "," );
	    strcat(( char * ) &newname_s, dsi->searchBase_ );
	    }

	//
	// Create the modification array
	//
	midx	= 0;
	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    mod_attr	= malloc( sizeof( LDAPMod ));
	    memset( mod_attr, 0, sizeof( LDAPMod ));

	    attribute	= bas_a[ idx++ ];

	    mod_attr->mod_op	= LDAP_MOD_REPLACE;
 	    mod_attr->mod_type	= attribute->t_val;

	    values	= ( char ** ) calloc( 50 + 1, sizeof( char * ));
	    vidx	= 0;

	    values[ vidx++ ]	= attribute->v_val;

	    values[ vidx ]	= NULL;

	    mod_attr->mod_vals.modv_strvals	= values;	// ????

	    attr_array[ midx++ ]	= mod_attr;
	    }					// while ( bas_a[ ] )

	attr_array[ midx ]	= NULL;

	stat	= ldap_modify_s(	dsi->ldapCtx_,
					( char * ) &newname_s,
					attr_array );

	dsi->ldapStatus_	= RS_LDAP_OK;

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_ATTR_MOD_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Modification of attribute(s) for '%s' failed, not completed, error %d (%s)", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    ret_status	= FALSE;
	    }

	idx	= 0;
	while ( attr_array[ idx ] != NULL )
	    {
	    free( attr_array[ idx ] );
	    attr_array[ idx ]	= NULL;
	    }					// while( attr_array[ idx ] != NULL )

	free( attr_array );

	return	ret_status;
	}					// End of ndi_ldap_modifyAttrByDN( )

//
//	ndi_ldap_removeAttrByDN
//
//	Remove an attribute in an entry specified by a DN and a list of
//	attributes to remove.The selected attributes must exists
//
//	Args:	nam_s	String with a DN without the defined searchbase
//		bas_a	AttributeInfo structure containing attributes
//			to be removed
//
//	Return:	TRUE	Entry was removed
//		FALSE	The entry was not removed
//
BOOLEAN ndi_ldap_removeAttrByDN( NamingContext *dsi, const char *nam_s, AttributeInfo **bas_a )
	{
int		stat		= -1;
int		midx		= 0;
int		idx		= 0;
int		vidx		= 0;

BOOLEAN		ret_status	= TRUE;

AttributeInfo	*attribute	= NULL;

LDAPMod		**attr_array	= NULL;
LDAPMod		*mod_attr	= NULL;

char		**values	= NULL;

DN_Area	newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	FALSE;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	attr_array	= ( LDAPMod ** ) calloc(( strlen(( char * ) *bas_a ) / sizeof( char * )) + 1, sizeof( char *  ));

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy(( char * ) &newname_s, nam_s );
	    strcat(( char * ) &newname_s, "," );
	    strcat(( char * ) &newname_s, dsi->searchBase_ );
	    }

	//
	// Create the modification array
	//
	midx	= 0;
	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    mod_attr	= malloc( sizeof( LDAPMod ));
	    memset( mod_attr, 0, sizeof( LDAPMod ));

	    attribute	= bas_a[ idx++ ];

	    mod_attr->mod_op	= LDAP_MOD_DELETE;
 	    mod_attr->mod_type	= attribute->t_val;

	    mod_attr->mod_vals.modv_strvals	= NULL;

	    attr_array[ midx++ ]	= mod_attr;
	    }					// while ( bas_a[ ] )

	attr_array[ midx ]	= NULL;

	stat	= ldap_modify_s(	dsi->ldapCtx_,
					( char * ) &newname_s,
					attr_array );

	dsi->ldapStatus_	= RS_LDAP_OK;

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_ATTR_REM_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Removal of attribute(s) for '%s' failed, not completed, error %d (%s) ", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    ret_status	= FALSE;
	    }

	idx	= 0;
	while ( attr_array[ idx ] != NULL )
	    {
	    mod_attr	= attr_array[ idx ];

	    free( mod_attr->mod_type );
//	    free( mod_attr->mod_vals.modv_strvals );

	    free( mod_attr );
	    attr_array[ idx++ ]	= NULL;
	    }					// while( attr_array[ idx ] != NULL )

	free( attr_array );

	return	ret_status;
	}					// End of ndi_ldap_removeAttrByDN( )

//
//	ndi_ldap_addAttrByDN
//
//	Add an attribute or attributes to an entry specified by a DN and
//	a list of Attributes. The selected attributes must NOT exists
//
//	NOTE ! Partial insert is possible
//
//	Args:	nam_s	String with a DN without the defined searchbase
//		bas_a	AttributeInfo structure containing attributes
//			to add for the entry
//
//	Return:	TRUE	Attributes was added
//		FALSE	The attributes was not added
//
BOOLEAN ndi_ldap_addAttrByDN( NamingContext *dsi, const char *nam_s, AttributeInfo **bas_a )
	{
int		stat		= -1;
int		midx		= 0;
int		idx		= 0;
int		vidx		= 0;

BOOLEAN		ret_status	= TRUE;

LDAPMod		**attr_array	= NULL;
LDAPMod		*mod_attr	= NULL;

char		**values	= NULL;

AttributeInfo	*attribute	= NULL;

DN_Area		newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	FALSE;
	    }

	attr_array	= ( LDAPMod ** ) calloc(( strlen(( char * ) *bas_a ) / sizeof( char * )) + 1, sizeof( char * ));

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy(( char * ) newname_s, nam_s );
	    strcat(( char * ) newname_s, "," );
	    strcat(( char * ) newname_s, dsi->searchBase_ );
	    }
	//
	// Create the modification array
	//
	midx	= 0;
	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    mod_attr	= malloc( sizeof( LDAPMod ));
	    memset( mod_attr, 0, sizeof( LDAPMod ));

	    attribute	= bas_a[ idx++ ];

	    mod_attr->mod_op	= LDAP_MOD_ADD;
 	    mod_attr->mod_type	= attribute->t_val;

	    values	= ( char ** ) calloc( 50 + 1, sizeof( char * ));
	    vidx	= 0;

	    values[ vidx++ ]	= attribute->v_val;

	    values[ vidx ]	= NULL;

	    mod_attr->mod_vals.modv_strvals	= values;	// ????

	    attr_array[ midx++ ]	= mod_attr;
	    }					// while ( bas_a[ ] )

	attr_array[ midx ]	= NULL;

	stat	= ldap_modify_s(	dsi->ldapCtx_,
					( char * ) &newname_s,
					attr_array );

	dsi->ldapStatus_	= RS_LDAP_OK;

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_ATTR_ADD_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Add of attribute(s) for '%s' failed, not completed, error %d (%s) ", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    ret_status	= FALSE;
	    }

	idx	= 0;
	while ( attr_array[ idx ] != NULL )
	    {
	    mod_attr	= attr_array[ idx ];

	    free( mod_attr->mod_type );
	    free( mod_attr->mod_vals.modv_strvals );
	    free( mod_attr );

	    attr_array[ idx++ ]	= NULL;
	    }					// while( attr_array[ idx ] != NULL )

	free( attr_array );

	return	ret_status;
	}					// End of ndi_ldap_addAttrByDN( )

//
//	ndi_ldap_getListByAttrs
//
//	Retrieves all attributes from an entry selected by the passed
//	attribute pairs.
//	It returns all attributes found ( except the password attribute )
//	If nothing found a NULL is returned
//
//	Args:	nam_s	String to add to searchbase, normaly len = 0
//		bas_a	AttributeInfo structure containing attributes
//			to search for
//
//	Return:		If found an AttributeInfo structure is returned
//			NULL if any error
//
AttributeInfo **ndi_ldap_getListByAttrs( NamingContext *dsi, const char *nam_s, AttributeInfo *bas_a[ ] )
	{

int		stat		= -1;
int		idx		= 0;
int		num_entries	= 0;

LDAPMessage	*result		= NULL;
LDAPMessage	*one_entry	= NULL;

char		**sea_a		= NULL;

char		*attr_name	= NULL;

BerElement	*ber_element	= NULL;
BerValue	**attr_value	= NULL;

AttributeInfo	*attribute	= NULL;

AttributeInfo	**res_a		= NULL;

char		*res_s		= NULL;

char		*filter		= NULL;

DN_Area		newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	NULL;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy(( char * ) &newname_s, nam_s );
	    strcat(( char * ) &newname_s, "," );
	    strcat(( char * ) &newname_s, dsi->searchBase_ );
	    }

	// Create the filter string

	filter	= malloc( FILTER_AREA_SIZE );

	memset( filter, 0, FILTER_AREA_SIZE );

	strcat( filter, "(&" );
	//
	// Add the attributes to filter on
	//
	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    attribute	= bas_a[ idx++ ];

	    strcat( filter, "(" );
	    strcat( filter, attribute->t_val );
	    strcat( filter, "=" );
	    strcat( filter, attribute->v_val );
	    strcat( filter, ")" );
	    }					// while ( bas_a[ ] )

	strcat( filter, ")" );

	// And finaly do the search

	stat = ldap_search_s( dsi->ldapCtx_,
				dsi->searchBase_,
				LDAP_SCOPE_SUBTREE,	// Scope
				filter,			// FILTER
				NULL,			// Attributes
				FALSE,			// Only types
				&result );		// Search result

	free( filter );

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attribute(s) for '%s' failed, error %d (%s) ", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    return NULL;
	    }

	// if result has more than one entry then something is wrong,
	// the arguments will not give ONE hit which is required

	num_entries = ldap_count_entries( dsi->ldapCtx_, result );

	if ( num_entries == 0 )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attributes for : '%s' failed, not found ", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return NULL;
	    }					// If ( num_entries == 0 )

	if ( num_entries > 1 )
	    {
	    // If there exits more elements, then return NULL since it was a bad match

	    dsi->ldapStatus_	= RS_LDAP_SEARCH_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Search returned ambiguous result of : '%s' - more than one match", newname_s );

	   if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }

	// OK, check the result

	one_entry	= ldap_first_entry( dsi->ldapCtx_, result );

	attr_name	= ldap_first_attribute( dsi->ldapCtx_, one_entry, &ber_element );

	// OK, loop and get the attribute name and value
	// But first allocate a return structure
	//
	// Allocate enough room for a pointer to the return attribute, plus
	// a NULL terminating pointer. 
	//
	res_a	= ( AttributeInfo ** ) calloc(( MAX_ATTRIBUTES + 1), sizeof( AttributeInfo * ));

	idx	= 0;
	while ( attr_name != NULL )
	    {
	    attribute	= malloc( sizeof( AttributeInfo ));
	    memset( attribute, 0, sizeof( AttributeInfo ));

	    res_a[ idx++ ]	= attribute;

	    attribute->t_val	= strdup( attr_name );

	    attr_value	= ldap_get_values_len( dsi->ldapCtx_, one_entry, attr_name );

	    // Check if attribute has a value, the password attribute
	    // does not have one, then return an empty object, not NULL

	    if ( attr_value != NULL )
		{
		res_s	= malloc(( attr_value[ 0 ]->bv_len ) + 2 );
		memcpy( res_s, attr_value[ 0 ]->bv_val, attr_value[ 0 ]->bv_len );

		attribute->v_len	= attr_value[ 0 ]->bv_len;
		attribute->v_val	= res_s;
		}				// if ( attr_value == NULL )

	    ldap_memfree( attr_name );

	    attr_name	= NULL;

	    ldap_value_free_len( attr_value );

	    attr_value	= NULL;

	    attr_name	= ldap_next_attribute( dsi->ldapCtx_, one_entry, ber_element );
	    }					// while ( attr_name != NULL )

	dsi->ldapStatus_	= RS_LDAP_OK;

	if ( result != NULL )
	    {
	    ldap_msgfree( result );
	    }  

	if ( ber_element != NULL )
	    {
	    ber_free( ber_element, 0 );
	    }					// if ( ber_element != NULL )

	return	res_a;
	}					// End of ndi_ldap_getListByAttrs(  );

//
//	ndi_ldap_getAttrValueByAttr
//
//	Returns the value of a selected attribute from an antry specified by
//	a list of attribute pairs.
//
//	Returns the specified attribute value string if found else NULL
//
//	NOTE ! Works only with nonbinary values
//
//	Args:	nam_s	String to add to the searchBase, normaly len = 0
//		bas_a	AttributeInfo structure containing attributes
//			specifying the search.
//		attr_s	The attribute value to return
//
//	Return:		Pointer to an attribute value string
//			NULL if any error
//
char *ndi_ldap_getAttrValueByAttr( NamingContext *dsi, const char *nam_s, AttributeInfo *bas_a[ ], const char *attr_s )
	{

int		idx		= 0;
int		stat		= -1;
int		num_entries	= 0;

LDAPMessage	*result		= NULL;
LDAPMessage	*one_entry	= NULL;

char		**sea_a		= NULL;

char		*attr_name	= NULL;

AttributeInfo	*attribute	= NULL;

BerElement	*ber_element	= NULL;
BerValue	**attr_value	= NULL;

char		*res_s		= NULL;

char		*filter		= NULL;

DN_Area	newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	NULL;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	//
	// Allocate enough room for a pointer to the attribute, plus
	// a NULL terminating pointer. 
	//

	sea_a	= ( char ** ) calloc(( 1 + 1), sizeof( char * ) );

	sea_a[ 0 ]	= strdup( attr_s );

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy(( char * ) &newname_s, nam_s );
	    strcat(( char * ) &newname_s, "," );
	    strcat(( char * ) &newname_s, dsi->searchBase_ );
	    }

	// Create the filter string

	filter	= malloc( FILTER_AREA_SIZE );

	memset( filter, 0, FILTER_AREA_SIZE );

	strcat( filter, "(&" );
	//
	// Add the attributes to filter string
	//
	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    attribute	= bas_a[ idx++ ];

	    strcat( filter, "(" );
	    strcat( filter, attribute->t_val );
	    strcat( filter, "=" );
	    strcat( filter, attribute->v_val );
	    strcat( filter, ")" );
	    }					// while ( bas_a[ ] )

	strcat( filter, ")" );

	// And finaly do the search

	stat = ldap_search_s( dsi->ldapCtx_,
				dsi->searchBase_,
				LDAP_SCOPE_SUBTREE,	// Scope
				filter,			// Filter
				sea_a,			// Attributes
				FALSE,			// Only types
				&result );		// Search result

	free( filter );

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_SEARCH_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attribute value for '%s' using attributes failed, error %d (%s) ", newname_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    return NULL;
	    }

	// if nat has more than one argument then something is wrong,
	// the arguments will not give ONE hit which is required

	num_entries = ldap_count_entries( dsi->ldapCtx_, result );

	if ( num_entries == 0 )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( "Retrieve of attribute value for '%s' using attributes in : '%s' failed", attr_s, newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return NULL;
	    }					// If ( num_entries == 0 )

	if ( num_entries > 1 )
	    {
	    // If there exits more elements, then return NULL since it was a bad match

	    dsi->ldapStatus_	= RS_LDAP_SEARCH_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Search returned ambiguous result of : '%s' - more than one match", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }

	// OK, check the result

	one_entry	= ldap_first_entry( dsi->ldapCtx_, result );

	attr_name	= ldap_first_attribute( dsi->ldapCtx_, one_entry, &ber_element );

	if ( attr_name == NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_NO_ATTR;

	    sprintf( dsi->ldapStatusMsg1_, "The specified attribute was not found for : '%s' - select attribute is '%s'", newname_s, attr_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }					// if ( attr_name == NULL )

	attr_value	= ldap_get_values_len( dsi->ldapCtx_, one_entry, attr_name );

	// Check if attribute has a value, the password attribute
	// does not have one, then return an empty object, not NULL

	res_s	= malloc(( attr_value[ 0 ]->bv_len ) + 2 );
	memset( res_s, 0, ( attr_value[ 0 ]->bv_len ) + 2 );

	if ( attr_value != NULL )
	    {
	    memcpy( res_s, attr_value[ 0 ]->bv_val, attr_value[ 0 ]->bv_len );
	    }				// if ( attr_value == NULL )

	dsi->ldapStatus_	= RS_LDAP_OK;

	ldap_memfree( attr_name );

	attr_name	= NULL;

	attr_name	= ldap_next_attribute( dsi->ldapCtx_, one_entry, ber_element );

	// if we now have more attributes then we are in trouble, return
	// a NULL object instead

	if ( attr_name != NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_ATTR_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Ambiguous attributes returned for : '%s' - select attribute is '%s'", newname_s, attr_s );

	    free( res_s );

	    res_s	= NULL;
	    }

	ldap_value_free_len( attr_value );

	if ( result != NULL )
	    {
	    ldap_msgfree( result );
	    }  

	if ( ber_element != NULL )
	    {
	    ber_free( ber_element, 0 );
	    }					// if ( ber_element != NULL )

	return	res_s;
	}					// End of ndi_ldap_getAttrValueByAttr(  )

//
//	ndi_ldap_getAttrByAttr
//
//	Get a specified attribute from an entry selected by a list of attribute pairs.
//
//	Returns the specified attribute if found else NULL
//
//	Args:	nam_s	String to add to the search base, normaly len = 0
//		bas_a	AttributeInfo structure containing attributes
//			to specify the search
//		attr_s	The attribute to return
//
//	Return:		A pointer to an AttributeInfor structure
//			NULL if any error
//
AttributeInfo *ndi_ldap_getAttrByAttr( NamingContext *dsi, const char *nam_s, AttributeInfo **bas_a, const char *attr_s )
	{

int		stat		= -1;
int		idx		= 0;
int		num_entries	= 0;

LDAPMessage	*result		= NULL;
LDAPMessage	*one_entry	= NULL;

char		**sea_a	= NULL;

char		*attr_name	= NULL;

BerElement	*ber_element	= NULL;
BerValue	**attr_value	= NULL;

AttributeInfo	*attribute	= NULL;

char		*res_s		= NULL;

char		*filter		= NULL;

DN_Area		newname_s;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	NULL;
	    }

	memset(( char * ) &newname_s, 0, sizeof( DN_Area ));
	strcpy(( char * ) &newname_s, dsi->searchBase_ );

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	//
	// Allocate enough room for a pointer to the attribute, plus
	// a NULL terminating pointer. 
	//

	sea_a	= ( char ** ) calloc(( 1 + 1), sizeof( char * ) );

	sea_a[ 0 ]	= strdup( attr_s );

	if ( strlen( nam_s ) != 0 )
	    {
	    strcpy( newname_s, nam_s );
	    strcat(( char * ) newname_s, "," );
	    strcat(( char * ) newname_s, dsi->searchBase_ );
	    }

	// Create the filter string

	filter	= malloc( FILTER_AREA_SIZE );

	memset( filter, 0, FILTER_AREA_SIZE );

	strcat( filter, "(&" );
	//
	// Add the attributes to filter on
	//
	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    attribute	= bas_a[ idx++ ];

	    strcat( filter, "(" );
	    strcat( filter, attribute->t_val );
	    strcat( filter, "=" );
	    strcat( filter, attribute->v_val );
	    strcat( filter, ")" );
	    }					// while ( bas_a[ ] )

	strcat( filter, ")" );

	// And finaly do the search

	stat = ldap_search_s( dsi->ldapCtx_,
				dsi->searchBase_,
				LDAP_SCOPE_SUBTREE,	// Scope
				filter,			// Filter
				sea_a,			// Attributes
				FALSE,			// Only types
				&result );		// Search result

	free( filter );

	if ( stat != LDAP_SUCCESS )
	    {
	    dsi->ldapStatus_	= RS_LDAP_SEARCH_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attribute value for '%s' using attributes failed, error %d (%s) ", attr_s, stat, ldap_err2string( stat ));

	    ndi_ldap_get_optional_message( dsi );

	    return NULL;
	    }

	// if nat has more than one argument then something is wrong,
	// the arguments will not give ONE hit which is required

	num_entries = ldap_count_entries( dsi->ldapCtx_, result );

	if ( num_entries == 0 )
	    {
	    dsi->ldapStatus_	= RS_LDAP_GET_ATTR_FAIL;

	    sprintf( dsi->ldapStatusMsg1_, "Retrieve of attributes for : '%s' failed, not found", newname_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return NULL;
	    }					// If ( num_entries == 0 )

	if ( num_entries > 1 )
	    {
	    // If there exits more elements, then return NULL since it was a bad match

	    dsi->ldapStatus_	= RS_LDAP_SEARCH_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Search returned ambiguous result of : '%s' - more than one match", newname_s );

	   if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }

	// OK, check the result

	one_entry	= ldap_first_entry( dsi->ldapCtx_, result );

	attr_name	= ldap_first_attribute( dsi->ldapCtx_, one_entry, &ber_element );

	if ( attr_name == NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_NO_ATTR;

	    sprintf( dsi->ldapStatusMsg1_, "The specified attribute was not found for : '%s' - select attribute is '%s'", newname_s, attr_s );

	    if ( result != NULL )
		{
		ldap_msgfree( result );
		}  

	    return	NULL;
	    }					// if ( attr_name == NULL )

	attribute	= malloc( sizeof( AttributeInfo ));
	memset( attribute, 0, sizeof( AttributeInfo ));

	attribute->t_val	= strdup( attr_name );

	attr_value	= ldap_get_values_len( dsi->ldapCtx_, one_entry, attr_name );

	// Check if attribute has a value, the password attribute
	// does not have one, then return an empty object, not NULL

	if ( attr_value != NULL )
	    {

	    res_s	= malloc(( attr_value[ 0 ]->bv_len ) + 2 );
	    memcpy( res_s, attr_value[ 0 ]->bv_val, attr_value[ 0 ]->bv_len );

	    attribute->v_len	= attr_value[ 0 ]->bv_len;
	    attribute->v_val	= res_s;
	    }				// if ( attr_value == NULL )

	dsi->ldapStatus_	= RS_LDAP_OK;

	ldap_memfree( attr_name );

	attr_name	= NULL;

	attr_name	= ldap_next_attribute( dsi->ldapCtx_, one_entry, ber_element );

	// if we now have more attributes then we are in trouble, return
	// a NULL object instead

	if ( attr_name != NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_ATTR_AMBIGUOUS;

	    sprintf( dsi->ldapStatusMsg1_, "Ambiguous attributes returned for : '%s' - select attribute is '%s'", newname_s, attr_s );

	    ldap_memfree( attr_name );

	    free( attribute );

	    attribute	= NULL;
	    }

	ldap_value_free_len( attr_value );

	if ( result != NULL )
	    {
	    ldap_msgfree( result );
	    }  

	if ( ber_element != NULL )
	    {
	    ber_free( ber_element, 0 );
	    }					// if ( ber_element != NULL )

	return	attribute;
	}					// End of ndi_ldap_getAttrByAttr(  );

//
//	ndi_ldap_removeSearchBase
//
//      Since the DN returned by getDN... contains also the search base
//      and that should not be included when passing a DN as a base for
//      a search, this method removes the search base from the DN and also
//      the preceeding ","
//
//	Args:	dn_s	Pointer to the DB string containing the search base
//
//	Return:	*	A pointer to the DN without search base if found
//			else NULL;
//
char *ndi_ldap_removeSearchBase( NamingContext *dsi, const char *dn_s )
	{
char		*dn_end		= NULL;
char		*res_s		= NULL;

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;

	// Get storage area for the result

	res_s	= malloc( strlen( dn_s ) + 10 );
	memset( res_s, 0, strlen( dn_s ) + 10 );

	strcpy( res_s, dn_s );

	ndi_ldap_str_to_upper( res_s );

	// Get the position of the search base

	dn_end	= strstr( res_s, dsi->searchBase_ );

	if ( dn_end == NULL )
	    {
	    dsi->ldapStatus_	= RS_LDAP_OK;

	    free( res_s );

	    return		NULL;
	    }

	dn_end	= dn_end - 1;
	*dn_end	= 0;

	dsi->ldapStatus_	= RS_LDAP_OK;

	// Ok, return the DN string pointer

	return	res_s;
        }                                       // End of removeSearchBase( )

//
//	ndi_ldap_setFollowReferral
//
//	It will if TRUE set the referral flag to follow else it will
//	be set to "ignore" wich is the standard
//	The flag must be set to TRUE before performing writes or updates to
//	the directory
//
//	Args:	flag	If TRUE set to follow referrals
//			if FALSE set to NOT follow referrals
//
//	Return:	TRUE	Was set
//		FALSE	Was not set
//
BOOLEAN ndi_ldap_setFollowReferral( NamingContext *dsi, BOOLEAN flag )
	{
BOOLEAN	ret_status	= TRUE;

	// If not bound then just exit, keeping status as it was

	if ( ndi_ldap_isNotBound( dsi ))
	    {
	    return	FALSE;
	    }

	return	ret_status;
	}					// End of ndi_ldap_setFollowReferral( )

//
//	ndi_ldap_freeAttributeInfo
//
//	Free the attribute info structure
//
//	Args:	bas_a	AttributeInfo structure to free
//
//	Return:		Always success
//
void ndi_ldap_freeAttributeInfo( AttributeInfo **bas_a )
	{
int		idx		= 0;

AttributeInfo	*attribute	= NULL;

	idx	= 0;
	while ( bas_a[ idx ] != NULL )
	    {
	    attribute	= bas_a[ idx ];

	    if ( attribute != NULL )
		{
		if ( attribute->t_val != NULL )
		    {
		    free( attribute->t_val );
		    }

		if ( attribute->v_val != NULL )
		    {
		    free( attribute->v_val );
		    }

		attribute->t_val	= NULL;
		attribute->v_val	= NULL;
		attribute->v_len	= 0;

		free( attribute );

		bas_a[ idx++ ]	= NULL;
		}				// if ( attribute != NULL )
	    }					// while ( bas_a[ ] )

	free( bas_a );

	return;
	}					// End of ndi_ldap_freeAttributeInfo( )

//
//	ndi_ldap_getLDAPCtx
//
//	Returns a pointer to the internal LDAP context. To be used if the
//	DirectoryService package could not fullfill the requirements of
//	the user. This pointer is used when calling standard LDAP functions
//
//	Input:	NONE
//
//	Return:		Pointer to the internal LDAP context block.
//
//
LDAP	*ndi_ldap_getLDAPCtx( NamingContext *dsi )
	{
	return	dsi->ldapCtx_;
	}					// End of ndi_ldap_getLDAPCtx( )

//
//	ndi_ldap_getStatus
//
//	Returns the last set status code
//
//	Input:	NONE
//
//	Return:		The current internal status code ( RS_LDAP_.. )
//
int	ndi_ldap_getStatus( NamingContext *dsi )
	{
	return	dsi->ldapStatus_;
	}					// End of ndi_ldap_getStatus( )
//
//	ndi_ldap_isNotBound
//
//	Returns wether or not a LDAP server has been bound
//
//	Input:	NONE
//
//	Return:		TRUE if bound
//			FALSE if not bound
//
BOOLEAN	ndi_ldap_isNotBound( NamingContext *dsi )
	{
	return	dsi->Bound_ == FALSE;
	}					// End of ndi_ldap_isNotBound( )
//
//	ndi_ldap_getMessage1
//
//	Returns the current error message. Valid only if status is not RS_LDAP_OK
//	or RD_LDAP_UNDEFINED
//
//	Input:	NONE
//
//	Return:		Pointer to character string
//
//	NOTE !		Should not be freed
//
char *ndi_ldap_getMessage1( NamingContext *dsi )
	{
	return	dsi->ldapStatusMsg1_;
	}					// End of ndi_ldap_getMessage1( )
//
//	ndi_ldap_getMessage2
//
//	Returns the option error message. Valid only if status is not RS_LDAP_OK
//	or RD_LDAP_UNDEFINED
//
//	Input:	NONE
//
//	Return:		Pointer to character string if otional message is defined in LDAP
//
//	NOTE !		Should not be freed
//
char *ndi_ldap_getMessage2( NamingContext *dsi )
	{
	return	dsi->ldapStatusMsg2_;
	}					// End of ndi_ldap_getMessage2( )

//
//	ndi_ldap_get_optional_message	( INTERNAL ONLY )
//
//	Retrieves the LDAP internal error message if it exists
//
//	Input:	NONE
//
//	Return:	NONE
//
void ndi_ldap_get_optional_message( NamingContext *dsi )
	{
int	stat;
char	*errmsgp	= NULL;

	stat	= ldap_get_option( dsi->ldapCtx_, LDAP_OPT_ERROR_STRING, &errmsgp );

	if (( stat != -1 ) && ( strlen(errmsgp) != 0 ))
	    {
	    strcpy( dsi->ldapStatusMsg2_, errmsgp );
	    ldap_memfree( errmsgp );
	    }

	return;
	}					// End of ndi_ldap_get_optional_message( dsi )

//
//  FUNCTION:
//	
//	ndi_ldap_str_to_upper
//
//  FUNCTIONAL DESCRIPTION:
//
//	This routine converts a character string to uppercase. It is a copy
//	of the ge_str_to_upper( ) from GE
//	
//  PROCESSING:
//
//	Convert character by character to uppercase until NULL is found
//
//  FORMAL PARAMETERS:
//    
//	char  *str     Pointer to a null terminated string
//
//  RETURN VALUE:
//
//	none
//
//  CALLING SEQUENCE:
//   
//      
//  
//  AUTHOR:
//
//      Tom Svaleklev, SEB Data                    6-MAR-1991
// 
void ndi_ldap_str_to_upper( char *str)
{
int   i = 0;

   while (str[i])
	{
	if	((((UBYTE)str[i] >= (UBYTE) 97 ) && 
		  ((UBYTE)str[i] <= (UBYTE)122 )) ||
		 (((UBYTE)str[i] >= (UBYTE)224 ) &&
		  ((UBYTE)str[i] <= (UBYTE)253)))
	    {
	    str[i] = (char) (UBYTE) str[i] - (UBYTE) 32;
	    }
	i++;
	}
}						// End of ndi_ldap_str_to_upper( )

//
//	ndi_ldap_close
//
//	Will close all connections to the bound server. All internal
//	structures are freed and any call to the service after this call
//	is invalid and will return unpredictable results.
//
//	Input:	NONE
//
//	Return:	NONE
//
void	ndi_ldap_close( NamingContext *dsi )
	{

	ldap_unbind( dsi->ldapCtx_ );

	if ( dsi->ldapServer_ != NULL )
	    {
	    free( dsi->ldapServer_ );

	    dsi->ldapServer_ = NULL;
	    }

	if ( dsi->searchBase_ != NULL )
	    {
	    free( dsi->searchBase_ );

	    dsi->searchBase_	= NULL;
	    }

	if ( dsi->ldapStatusMsg1_ != NULL )
	    {
	    free( dsi->ldapStatusMsg1_ );
	    }

	if ( dsi->ldapStatusMsg2_ != NULL )
	    {
	    free( dsi->ldapStatusMsg2_ );
	    }

	dsi->ldapStatus_	= RS_LDAP_UNDEFINED;
	dsi->ldapCtx_		= NULL;
	dsi->Bound_		= FALSE;

	free( dsi );
	}					// End of ndi_ldap_close( )
